% Automatic detection of amalgamation in landslide inventories
% by Odin Marc. Created November 2014.

% This script is associated with a manuscript publish in NHESS. 
% The nalgorithm is described and benchmarked in it.
% If you publish work based on this algorithm or a modified version of it
% please cite the NHESS paper: 
% Nat. Hazards Earth Syst. Sci. Discuss., 2, 7651-7678, 2014
% www.nat-hazards-earth-syst-sci-discuss.net/2/7651/2014/
% doi:10.5194/nhessd-2-7651-2014

%INPUT
% 3 files are required:
% 1: a tiff image containing the raster of landslide with their ID
% 2: a tiff image of a DEM raster corresponding to the whole area affected
% by landslide.
% NB: The two rasters must have exactly the same resolution and the same
% dimension so that they are converted into matrix with the same index.

% 3: a text file with 3 columns containing for each landslide its ID, Area
% and K value. K may be computed with Area, A and Perimeter P from:

% K = 0.5* ( 4 * (P/sqrt(pi*A) +1)^2 /9 -2 + sqrt( (4* (P/sqrt(pi*A) +1)^2 /9 -2)^2 - 4 )  )
% See the associated publication for more detail.

% OUTPUT
%If no errors occur two files will be exported:
% 1/ a tif image containing the skeleton of the polygons named :
% ['SkMap',Place,num2str(Ac),'_RBC',num2str(RBc),'_Sc',num2str(Sc),'.tif']
% 2/ a txt file containing the ID and result class of each analyzed polygon.
% where 0 means correct , N>0 is the number of branches, -1 means slope
% inconsistency,  -2 means ridge/channel crossing and -9 means failed analysis (normally rare). 



close all
clear all
% 
      file_L='File_Path.tif'; % TIF image of the landslide raster with ID value of the landslides in each cells
      file_dem='File_Path.tif'; % DEM raster in TIF format
      file_K='File_Path.txt'; % Id, Area, K values of the landslide inventory

Place='XXX'; %Inventory Name Code (for final file saving purpose)
reso=5 ;  % raster resolution, in m (Must be the same for both INPUT raster !!)

%User Parameters 
Ac=1e3;  % Minimal Landslide Area that will be analyzed.
Kc=2; % Minimal K value that will be analyzed. 
% For exhaustiveness, set Ac close at roll over size and Kc at 2. For fast
% but incomplete test, Ac and Kc may be increased

Sc=12; % minimal average slope in  of a landslide...  To be tuned between 10-15
RBc=5;  % every branches longer than the ratio of the main branch length, Lmax, over RBC will be considered wrong. To be tuned between 3-7.

%% loading data
%-------------------------------------
 IAK=load(file_K); I=IAK(:,1); A=IAK(:,2); K=IAK(:,3); Nt=length(I); %

 dem=geotiffread(file_dem); dem(dem<0)=0; dem(dem>1e4)=0; % load DEM and remove no data values

Amap=geotiffread(file_L); Amap(abs(Amap)>10*Nt)=0;  Amap=(double(Amap)); % load landslide ID map

%-------------------------------------

%First criterium: size limit 
%(Typically small polygons are numerous and with very simple shape, so they just slow down the algorithm without much results) 
 idKc=find(A > Ac); %Polygons smaller than Ac will not be considered


 %Main control loop
%--------------------------------------

Skmap=Amap; % create Skeleton Map. Polygon not analyzed will appear as filled polygons.

%idKc=find(AK==max(AK(:,1)));
N=length(idKc);
Amalgam=zeros(N,1);
for i=1:N;
    Ip=I(idKc(i));
 if K(idKc(i))> Kc
       [idPy,idPx]=find( Amap==Ip ); % index of given polygon
         if isempty(idPx)
           Amalgam(i)=-9;
         else
           imP=Amap(min(idPy):max(idPy),min(idPx):max(idPx));
       imP(imP~=Ip)=0;
%          if (max(max(idPy))-min(min(idPy)))*(max(max(idPx))-min(min(idPx))) > Ap*10 ; miss = miss+1; else

%% First test: Make skeleton and find if there are multiple branchs
%  skP=bwmorph(imP,'skel','Inf'); 
 thinP=bwmorph(imP,'thin','Inf'); % Thin avoid ford at extremities of circles/ellipses
    Skmap(min(idPy):max(idPy),min(idPx):max(idPx))=thinP;
%  figure(1), imagesc(imP/Ip +thinP),

bpP = bwmorph(thinP,'branchpoints') ; % find branching points (if any)

brBP=sum(sum(bpP)); branches=thinP ; 

% To display the landslide polygon with skeleton and branchpoints use the following command:  
%  figure(), imagesc(imP/Ip+thinP+bpP), 

   while brBP>0 % Loop breaking skeleton in branches

branches = branches & ~bpP; % set branch points to zero
bpP=bwmorph(branches,'branchpoints');
brBP=sum(sum(bpP));
   end  
   
[bLab,BrCut] = bwlabel( branches ); % label connected components
     BrLgth=zeros(1,BrCut);
     
    for j=1:BrCut %Loop computing the length of each branches
        BrLgth(j)=length(bLab(bLab==j));
    end  
        
    iLongLab=find(BrLgth==max(BrLgth)); %Finding index of the longest branch
        Lmax=max(BrLgth); % length of the longest branch
        RBL=sort(BrLgth./Lmax); % ratio of branches length over the longest branch
        if length(RBL)>1
               Amalgam(i)=length(RBL(RBL > 1/RBc)); %polygon attributed with the number of branches longer than Lmax/RBc
       
        else Amalgam(i)=0; % If only one long branches and small appendix the polygon is not consider amalgamated yet
        
        end
   
   if Amalgam(i) == 0 %i.e. polygon without multiple branches may have slope inconsitencies or ridge/channel crossing ->  compare with DEM
%% Second part: check consistency of polygon with DEM
       
%Load the elevation within the landslide polygon       
demP=dem(min(idPy):max(idPy),min(idPx):max(idPx));  demP(imP~=Ip)=0;
demPsk=demP; demPsk(thinP==0)=0;

% To display the elevation within a landslide polygon, use following command   
% figure(), imagesc(demP+demBr)    

if sum(sum(demPsk))>0

 demBr=demPsk; demBr((bLab~=iLongLab))=0; %get DEM of the main branch

 % Check if any cell of the main branche is higher or lower than the endpoints -> likely ridge/river crossing
 Endp=demBr(bwmorph(demBr,'endpoints'));
 if isempty(Endp), Amalgam(i)=-2; else  
   if max(max(demBr))>max(Endp) || min(min(demBr))>min(Endp) 
            Amalgam(i)=-2;
   end
 end
 
   %Check if the elevation change is consistent with a minimal slope
 DHBr=max(max(demBr))-min(min(demBr)); %elevation change along the main branch
 minDrop=tand(Sc)*length(demBr(demBr>0))*reso; % Expected min drop for a given critical slope Sc

  if  DHBr<minDrop   %i.e. likely to be a grouping of parallel  slides that appear perpendicular to slope now
     Amalgam(i)=-1;
  end
  
else Amalgam(i)=-9;
end
   end 
      
         end
         
         
 
 end
 
 
 
end
 
%% file export
imwrite(Skmap,['SkMap',Place,num2str(Ac),'_RBC',num2str(RBc),'_Sc',num2str(Sc),'_Kc',num2str(Kc),'.tif'],'tif')

idMulti=find(Amalgam>=1); % polygon with multiple branches detected
idCor= find(Amalgam==0); % polygon classified as correct
idSlope= find(Amalgam==-1); % polygon with slope inconsistency detected
idRidge= find(Amalgam==-2); % polygons with ridge or channel crossing detected



Tosave=[I(idKc),Amalgam];
FID=fopen(['ID_Amalgam',Place,num2str(Ac),'_RBC',num2str(RBc),'_Sc',num2str(Sc),'_Kc',num2str(Kc),'.txt'],'w');
fprintf(FID,'%i %i\n',Tosave');
fclose(FID)